# Based on template for mkClient usage with app-specific network messages
extends Node
class_name ClientNode

const DEBUG_OUTPUT = true

var _networkClient:mkClient = null
var _clients:Dictionary[int, Avatar] = {} # Includes local client avatar

#region Client custom messages definition
enum myMsgId {
	ID_CHATLINE_S,  ## Network message ID. Shall contain [string]
	ID_SET_TO_II,   ## Network message ID. Shall contain [int, int]
	ID_MOVE_TO_II,  ## Network message ID. Shall contain [int, int]
}

var myMsgHandlers:Dictionary[int, Callable] = {
	myMsgId.ID_CHATLINE_S: _handle_incoming_ID_CHATLINE_S,
	myMsgId.ID_MOVE_TO_II: _handle_incoming_ID_MOVE_TO_II,
	myMsgId.ID_SET_TO_II: _handle_incoming_ID_SET_TO_II
}
#endregion

#region Network client exposed signals
func _onSIG_ConnectionChanged(_newState:WebSocketPeer.State):
	if DEBUG_OUTPUT:
		print("mkClient_Run: _onSIG_ConnectionChanged called. State is now " + _networkClient.GetConnectionStateAsString()) 
	SetStatusText()

func _onSIG_MyClientIdReceived(myId:int):
	if DEBUG_OUTPUT:
		print("mkClient_Run: _onSIG_MyClientIdReceived called. My Id is now " + _networkClient.GetClientIdInRoomAsString())
	SetStatusText()
	_createAvatar(myId, true)
	Send_ID_SET_TO_II(0, 0)
	
func _onSIG_NewClientInRoom(newId:int):
	if DEBUG_OUTPUT:
		print("mkClient_Run: _onSIG_NewClientInRoom called with newId " + str(newId))
	_createAvatar(newId, false)
	var pixelPos = _clients[_networkClient.GetClientIdInRoom()].GetAvatarPosition()
	var indexPos = Playfield.Instance.GetFieldIndexFromPixelPos(pixelPos.x, pixelPos.y)
	Send_ID_SET_TO_II(indexPos.x, indexPos.y)
	
func _onSIG_DataReceived(fromClientId:int, payload:PackedByteArray):
	if DEBUG_OUTPUT:
		print("mkClient_Run: _onSIG_DataReceived called with raw data:")
		print(payload)
	# A client with given id (in our room) sent a data message. We now need to
	# evaluate this (custom data) and take actions
	var retrievedData = mkAppMsg.DeconstructMsg(payload)
	if (retrievedData.size() == 0):
		print("mkClient_Run: Could not deconstruct received data bytes")
		return
	else:
		print("mkClient_Run: Deconstructed message from data bytes is")
		print(retrievedData)
		
	if (myMsgHandlers.has(retrievedData[0])):
		# Invoke the predefined message handler for this custom msgId 
		myMsgHandlers[retrievedData[0]].call(fromClientId, retrievedData)
	else:
		print("mkClient_Run: Did not find this msg id to handle: " + str(retrievedData[0]))
	
func _onSIG_ClientLeftRoom(oldId:int):
	if DEBUG_OUTPUT:
		print("mkClient_Run: _onSIG_ClientLeftRoom called with oldId " + str(oldId))
	if not _clients.has(oldId):
		return
	_clients[oldId].queue_free()
	_clients.erase(oldId)
#endregion

#region Client custom message handlers (incoming)
func _handle_incoming_ID_CHATLINE_S(_fromClientId:int, _data:Array):
	if DEBUG_OUTPUT:
		print("mkClient_Run: _handle_incoming_ID_CHATLINE_S called with data:")
		print(_data)
	AddToChatWindow(_fromClientId, _data[1]) # _data[0] is the custom msg id

func _handle_incoming_ID_SET_TO_II(_fromClientId:int, _data:Array):
	if DEBUG_OUTPUT:
		print("mkClient_Run: _handle_incoming_ID_SET_TO_II called with data:")
		print(_data)
	if not _clients.has(_fromClientId):
		return
	if (_fromClientId == _networkClient.GetClientIdInRoom()):
		print("mkClient_Run: This is a message sent to the server by OUR client")
	else:
		print("mkClient_Run: This is a message sent to the server by ANOTHER client")

	_clients[_fromClientId].SetAvatarVisible(true)
	_positionAvatar(_clients[_fromClientId], _data[1], _data[2]) # _data[0] is the custom msg id
#endregion

func _handle_incoming_ID_MOVE_TO_II(_fromClientId:int, _data:Array):
	if DEBUG_OUTPUT:
		print("mkClient_Run: _handle_incoming_ID_MOVE_TO_II called with data:")
		print(_data)
	if not _clients.has(_fromClientId):
		return
	if (_fromClientId == _networkClient.GetClientIdInRoom()):
		print("mkClient_Run: This is a message sent to the server by OUR client")
	else:
		print("mkClient_Run: This is a message sent to the server by ANOTHER client")

	_moveAvatar(_clients[_fromClientId], _data[1], _data[2]) # _data[0] is the custom msg id
#endregion

#region Client custom message handlers (outgoing)
func Send_ID_CHATLINE_S(s1:String)->void:
	var payload = mkAppMsg.ConstructMsg(myMsgId.ID_CHATLINE_S, [s1])
	_networkClient.SendPayload(payload)
	if DEBUG_OUTPUT:
		print("mkClient_Run: func Send_ID_CHATLINE_S called with data:")
		print([s1])

func Send_ID_SET_TO_II(i1:int, i2:int)->void:
	var payload = mkAppMsg.ConstructMsg(myMsgId.ID_SET_TO_II, [i1, i2])
	_networkClient.SendPayload(payload)
	if DEBUG_OUTPUT:
		print("mkClient_Run: Send_ID_SET_TO_II called with data:")
		print([i1, i2])
		
func Send_ID_MOVE_TO_II(i1:int, i2:int)->void:
	var payload = mkAppMsg.ConstructMsg(myMsgId.ID_MOVE_TO_II, [i1, i2])
	_networkClient.SendPayload(payload)
	if DEBUG_OUTPUT:
		print("mkClient_Run: Send_ID_MOVE_TO_II called with data:")
		print([i1, i2])
#endregion

# Private Functions
func _ready():
	if DEBUG_OUTPUT:
		print("mkClient_Run: _ready called.")
		
	# Create the mkClient object that handles connection to the server
	_networkClient = mkClient.new()
	self.add_child(_networkClient)
	
	# Connect to signals exposed from mkClient
	_networkClient.SIG_ConnectionChanged.connect(_onSIG_ConnectionChanged)
	_networkClient.SIG_MyClientIdReceived.connect(_onSIG_MyClientIdReceived)
	_networkClient.SIG_NewClientInRoom.connect(_onSIG_NewClientInRoom)
	_networkClient.SIG_DataReceived.connect(_onSIG_DataReceived)
	_networkClient.SIG_ClientLeftRoom.connect(_onSIG_ClientLeftRoom)
	
	var sendButton = $ChatLine/ChatSendButton as Button
	sendButton.pressed.connect(SendButtonPressed)

func _input(event):
	if event is InputEventMouseButton:
		var iemb = event as InputEventMouseButton
		if (iemb.button_index != MOUSE_BUTTON_LEFT) or (event.is_pressed() == false):
			return
		var mousePos = iemb.position
		if (Playfield.Instance.IsPixelPosInPlayfield(mousePos.x, mousePos.y) == false):
			return
		var targetField = Playfield.Instance.GetFieldIndexFromPixelPos(mousePos.x, mousePos.y)
		Send_ID_MOVE_TO_II(targetField.x, targetField.y)

func _createAvatar(clientId:int, isLocal:bool)->void:
	if _clients.has(clientId):
		return
	var avatarPrefab = preload("res://Avatar/Avatar.tscn")
	var newClient = avatarPrefab.instantiate()
	add_child(newClient)
	_clients[clientId] = newClient
	_clients[clientId].SetIdLabel(str(clientId))
	_clients[clientId].SetAvatarColor(clientId)
	_clients[clientId].SetIsLocalClient(isLocal)
	_clients[clientId].SetAvatarVisible(false)
	# initial positioning will be done by a dedicated message sent by the client

func _positionAvatar(avatar:Avatar, mapIndexX:int, mapIndexY:int)->void:
	var targetPos = Playfield.Instance.GetFieldPixelPosFromIndex(mapIndexX, mapIndexY)
	avatar.SetAvatarPosition(targetPos.x, targetPos.y)

func _moveAvatar(avatar:Avatar, mapIndexX:int, mapIndexY:int)->void:
	var targetPos = Playfield.Instance.GetFieldPixelPosFromIndex(mapIndexX, mapIndexY)
	avatar.SetAvatarTargetPosition(targetPos.x, targetPos.y)

# Public Functions
func SetStatusText():
	var statusLabel = $StatusLabel as Label
	statusLabel.text = "Status: My Client Id = " + _networkClient.GetClientIdInRoomAsString() + ", Connection State = " + _networkClient.GetConnectionStateAsString()

func AddToChatWindow(_fromId:int, chatText:String):
	var chatWindow = $ChatWindow/ChatWindowText as RichTextLabel
	chatWindow.add_text("\n")
	chatWindow.add_text(chatText)

func SendButtonPressed()->void:
	var editLine = $ChatLine/ChatLineEdit as LineEdit
	if (editLine.text.length() == 0):
		return
	var idText = "Client " + _networkClient.GetClientIdInRoomAsString() + ": "
	Send_ID_CHATLINE_S(idText + editLine.text)
	editLine.clear()
