## The mkRelay Client. See the template mkClient_Run for information on how to handle a client.
## Configuration parameters are stored in 'mkClientConfig.gd'. Adapt those as you see fit.
extends Node
class_name mkClient

# ----- Constants & Signals
const NO_CLIENTID = 0
static var _wspStatesString = {
	WebSocketPeer.STATE_CLOSED:     "STATE_CLOSED",
	WebSocketPeer.STATE_CLOSING:    "STATE_CLOSING",
	WebSocketPeer.STATE_CONNECTING: "STATE_CONNECTING",
	WebSocketPeer.STATE_OPEN:       "STATE_OPEN"
}
signal SIG_ConnectionChanged(newState:WebSocketPeer.State) ## Called with new server connection state after it changed
signal SIG_MyClientIdReceived(myId:int) ## Called when we received our own client Id in the requested room
signal SIG_NewClientInRoom(newId:int)   ## Called when another client entered the room
signal SIG_ClientLeftRoom(oldId:int)    ## Called when another client left the room
signal SIG_DataReceived(fromId:int, payload:PackedByteArray) ## Called when this client received new data to process

# ----- Variables
var _socket:WebSocketPeer = null 
var _socketTLS:TLSOptions = null
var _oldSocketState:WebSocketPeer.State = WebSocketPeer.State.STATE_CLOSED
var _myIdInRoom:int = NO_CLIENTID

# ----- Private functions
func _ready()->void:
	if (mkClientConfig.PRINT_DEBUG_OUTPUT):
		print("mkClient: Creating and starting a new client")
	_createTLSOptions()
	_socket = WebSocketPeer.new()
	_connectToServer()
	
func _process(_delta:float)->void:
	_update()

func _createTLSOptions()->void:
	var client_certs:X509Certificate
	if mkClientConfig.USE_DEV_TLS:
		client_certs = load(mkRelayDevCerts.CERT_FILE) as X509Certificate
		_socketTLS = TLSOptions.client(client_certs)
	else:
		_socketTLS = TLSOptions.client_unsafe()

func _connectToServer()->void:
	if (mkClientConfig.PRINT_DEBUG_OUTPUT):
		print("mkClient: ConnectToServer called")
	if (!_socket) or (_socket.get_ready_state() != WebSocketPeer.STATE_CLOSED):
		if (mkClientConfig.PRINT_DEBUG_OUTPUT):
			print("mkClient: Socket not there or not in state closed")
		return
	var connectionString:String = str(mkClientConfig.SERVER_URL, ":" ,mkClientConfig.SERVER_LISTENING_PORT)
	_socket.connect_to_url(connectionString, _socketTLS)
	if (mkClientConfig.PRINT_DEBUG_OUTPUT):
		print("mkClient: Trying to connect to server " + connectionString)

func _onClientConnected()->void:
	if (mkClientConfig.PRINT_DEBUG_OUTPUT):
		print("mkClient: Connection with server established. Sending initial room request msg.")
	var msg:mkRelayMsg = mkRelayMsg.new()
	msg.CreateMsg_CS_REQ(mkClientConfig.ROOM_UUID) 
	_socket.send(msg.GetMsgAsBytes())
	
func _onClientDisconnected()->void:
	var code = _socket.get_close_code()
	var reason = _socket.get_close_reason()
	if (mkClientConfig.PRINT_DEBUG_OUTPUT):
		print("mkClient: WebSocket closed with code: %d, reason %s." % [code, reason])

func _checkIncomingData()->void:
	if (!_socket):
		return
	
	while _socket.get_available_packet_count():
		var msg:mkRelayMsg = mkRelayMsg.new()
		msg.CreateMsgFromBytes(_socket.get_packet())

		match msg.GetMsgType():
			mkRelayMsg.eNW_RELAY_MSG.SC_ACK:
				if (msg.GetIdInRoom() != NO_CLIENTID):
					_myIdInRoom = msg.GetIdInRoom()
					SIG_MyClientIdReceived.emit.call_deferred(msg.GetIdInRoom())
					if (mkClientConfig.PRINT_DEBUG_OUTPUT):
						print ("mkClient: We received our clientId (%d) in the room" % [_myIdInRoom])
			mkRelayMsg.eNW_RELAY_MSG.SC_NEW:
				if (msg.GetIdInRoom() != NO_CLIENTID):
					SIG_NewClientInRoom.emit.call_deferred(msg.GetIdInRoom())
					if (mkClientConfig.PRINT_DEBUG_OUTPUT):
						print ("mkClient: We were informed about another client (%d) in the room" % [msg.GetIdInRoom()])
			mkRelayMsg.eNW_RELAY_MSG.SC_DAT:
				if (msg.GetIdInRoom() != NO_CLIENTID):
					SIG_DataReceived.emit.call_deferred(msg.GetIdInRoom(),msg.GetPayload())
					if (mkClientConfig.PRINT_DEBUG_OUTPUT):
						print ("mkClient: We received data from client (%d) in the room" % [msg.GetIdInRoom()])
						mkRelayMsg.PrintMsgContent(msg)
			mkRelayMsg.eNW_RELAY_MSG.SC_OLD:
				if (msg.GetIdInRoom() != NO_CLIENTID):
					SIG_ClientLeftRoom.emit.call_deferred(msg.GetIdInRoom())
					if (mkClientConfig.PRINT_DEBUG_OUTPUT):
						print ("mkClient: We were informed about a leaving client (%d) in the room" % [msg.GetIdInRoom()])
			mkRelayMsg.eNW_RELAY_MSG.NW_ERR:
				if (mkClientConfig.PRINT_DEBUG_OUTPUT):
					print ("mkClient: ERROR, could not interpret server message")

func _update()->void:
	if (!_socket):
		return
		
	_socket.poll()
	
	var curSocketState = _socket.get_ready_state()
	var stateChanged:bool = (_oldSocketState != curSocketState)
	_oldSocketState = curSocketState
	
	if (stateChanged): # External signal
		SIG_ConnectionChanged.emit.call_deferred(curSocketState)
	
	match curSocketState:
		WebSocketPeer.STATE_CONNECTING:
			pass
		WebSocketPeer.STATE_OPEN:
			if (stateChanged):
				_onClientConnected()
			_checkIncomingData()
		WebSocketPeer.STATE_CLOSING:
			pass
		WebSocketPeer.STATE_CLOSED:
			if (stateChanged):
				_onClientDisconnected()

# ----- Public functions
func GetConnectionStateAsString()->String:
	return _wspStatesString[GetConnectionState()]

func GetConnectionState()->WebSocketPeer.State:
	if (!_socket):
		return WebSocketPeer.State.STATE_CLOSED
	return _socket.get_ready_state()

func GetClientIdInRoom()->int:
	return _myIdInRoom
	
func GetClientIdInRoomAsString()->String:
	if (_myIdInRoom == NO_CLIENTID):
		return "NO_CLIENTID"
	else:
		return str(_myIdInRoom)
	
func SendPayload(msgBytes:PackedByteArray):
	if (mkClientConfig.PRINT_DEBUG_OUTPUT):
		print("mkClient: SendPayload called")
	if (!_socket):
		if (mkClientConfig.PRINT_DEBUG_OUTPUT):
			print("mkClient: Socket not there")
		return
	if (_myIdInRoom == NO_CLIENTID):
		if (mkClientConfig.PRINT_DEBUG_OUTPUT):
			print("mkClient: Can't send. No clientId assigned from server yet")
		return
	if (_socket.get_ready_state() != WebSocketPeer.STATE_OPEN):
		if (mkClientConfig.PRINT_DEBUG_OUTPUT):
			print("mkClient: Can't send. Socket not in open state")
		return
	var msg:mkRelayMsg = mkRelayMsg.new()
	msg.CreateMsg_CS_DAT(_myIdInRoom, msgBytes)
	if (msg.GetMsgType() == mkRelayMsg.eNW_RELAY_MSG.NW_ERR):
		if (mkClientConfig.PRINT_DEBUG_OUTPUT):
			print("mkClient: ERROR creating data msg for sending to server")
		return
	_socket.send(msg.GetMsgAsBytes())
	if (mkClientConfig.PRINT_DEBUG_OUTPUT):
			print("mkClient: Sent message:")
			print(msg.GetMsgAsBytes())
