class_name mkRelayMsg

# The basic data wrapper for messages mkRelay Client<->Server

# Message Format:
# MSG_TYPE:       1 Byte
# ID_IN_ROOM:     1 Byte
# PAYLOAD:        x Bytes (limited by server)

# ----- Constants
const NO_ROOMID = 0
const FIXED_DATA_BYTES = 2
const UUID_STRING_LENGHT = 36

# Network messages 
enum eNW_RELAY_MSG 
{
	NW_ERR = 99,	# Used for error/invalid message
	
	# Network Msg ID Client->Server
	CS_REQ = 101, 	# New client sends request for Room
					# Data: UUID as string
	CS_DAT = 102, 	# Sent client data, to be distributed to all in room
					# Data: ClientId, PayloadLength, Payload

	# Network Msg ID Server->Client
	SC_ACK = 201,	# Accepts a new client
					# Data: ClientId in requested room
	SC_NEW = 202,	# Informs other clients about a new client
					# Data: ClientId in requested room  
	SC_DAT = 204,	# Data from one client to all in room
					# Data: ClientId, PayloadLength, Payload
	SC_OLD = 205,	# Informs other clients about a client leave
					# Data: ClientId in assigned room
}

# ----- Variables
var _msgType:eNW_RELAY_MSG = eNW_RELAY_MSG.NW_ERR
var _idInRoom:int
var _payload:PackedByteArray

var _msgBytes:PackedByteArray

# ----- Private functions
func _init():
	_clearData()

func _clearData():
	_msgType = eNW_RELAY_MSG.NW_ERR
	_idInRoom = NO_ROOMID
	_payload.clear()
	_createMsgBytes()

func _createMsgBytes():
	_msgBytes.resize(FIXED_DATA_BYTES)
	_msgBytes.encode_u8(0, _msgType)
	_msgBytes.encode_u8(1, _idInRoom)
	_msgBytes.append_array(_payload)

func _getNWMsgTypeFromInt(intVal:int)->eNW_RELAY_MSG:
	var msgKey = eNW_RELAY_MSG.find_key(intVal)
	if (msgKey == null):
		return eNW_RELAY_MSG.NW_ERR
	else:
		return eNW_RELAY_MSG[msgKey]

# ----- Public functions
func SetMsgType(newType:eNW_RELAY_MSG):
	_msgType = newType
	_createMsgBytes()

func GetMsgType()->eNW_RELAY_MSG:
	return _msgType
	
func GetMsgType_Str(id:eNW_RELAY_MSG)->String:
	var enumAsText: String = eNW_RELAY_MSG.find_key(id)	
	return enumAsText

func GetIdInRoom()->int:
	return _idInRoom
	
func GetPayloadLengh()->int:
	return _payload.size()
	
func GetPayload()->PackedByteArray:
	return _payload
	
func GetPayloadAsASCII()->String:
	return _payload.get_string_from_ascii()

func GetMsgAsBytes()->PackedByteArray:
	return _msgBytes
	
func IsValid()->bool:
	if (_msgType == eNW_RELAY_MSG.NW_ERR):
		return false
	return true
	
func CreateMsg_CS_REQ(roomUUID:String): 
	if (roomUUID.length() != UUID_STRING_LENGHT):
		_clearData()
		return
	_msgType = eNW_RELAY_MSG.CS_REQ
	_idInRoom = NO_ROOMID
	_payload.resize(36)
	_payload = roomUUID.to_ascii_buffer()
	_createMsgBytes()

func CreateMsg_SC_NEW(clientId:int):
	_msgType = eNW_RELAY_MSG.SC_NEW
	_idInRoom = clientId
	_payload.clear()
	_createMsgBytes()

func CreateMsg_CS_DAT(clientId:int, payload:PackedByteArray):
	_msgType = eNW_RELAY_MSG.CS_DAT
	_idInRoom = clientId
	_payload = payload
	_createMsgBytes()
	
func CreateMsg_SC_ACK(clientId:int):
	_msgType = eNW_RELAY_MSG.SC_ACK
	_idInRoom = clientId
	_payload.clear()
	_createMsgBytes()
	
func CreateMsg_SC_OLD(clientId:int):
	_msgType = eNW_RELAY_MSG.SC_OLD
	_idInRoom = clientId
	_payload.clear()
	_createMsgBytes()

func CreateMsgFromBytes(byteArray:PackedByteArray):
	if (byteArray.size() < FIXED_DATA_BYTES):
		_clearData()
		return
	
	_msgType = _getNWMsgTypeFromInt(byteArray.decode_u8(0))
	if (_msgType == eNW_RELAY_MSG.NW_ERR):
		_clearData()
		return
	
	_msgBytes.resize(FIXED_DATA_BYTES)
	_idInRoom = byteArray.decode_u8(1)
	
	if (byteArray.size() > FIXED_DATA_BYTES):
		_payload = byteArray.slice(FIXED_DATA_BYTES)
	else:
		_payload.clear()
	
	_createMsgBytes()

static func PrintMsgContent(m:mkRelayMsg):
	print(m.GetMsgType(), "(", m. GetMsgType_Str(m.GetMsgType()), "), ",
		  m.GetIdInRoom(), ", " ,
		  m.GetPayload(),  ", ",
		  "Valid: ", m.IsValid())
